<?php
/**
 * Plugin Name: URL Redirect Checker
 * Plugin URI: https://foxco.net
 * Description: Check URL redirects by pasting a list or uploading a CSV file. Returns original URLs in column A and final destination URLs in column B. Use shortcode [redirect_checker] for frontend.
 * Version: 1.0.0
 * Author: Foxco.net
 * Author URI: https://foxco.net
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: url-redirect-checker
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('URL_REDIRECT_CHECKER_VERSION', '1.0.0');
define('URL_REDIRECT_CHECKER_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('URL_REDIRECT_CHECKER_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Main Plugin Class
 */
class URL_Redirect_Checker {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Enqueue scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        
        // AJAX handlers
        add_action('wp_ajax_urc_check_redirects', array($this, 'ajax_check_redirects'));
        add_action('wp_ajax_nopriv_urc_check_redirects', array($this, 'ajax_check_redirects'));
        add_action('wp_ajax_urc_process_csv', array($this, 'ajax_process_csv'));
        add_action('wp_ajax_nopriv_urc_process_csv', array($this, 'ajax_process_csv'));
        
        // Register shortcode
        add_shortcode('redirect_checker', array($this, 'render_shortcode'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'URL Redirect Checker',
            'Redirect Checker',
            'manage_options',
            'url-redirect-checker',
            array($this, 'render_admin_page'),
            'dashicons-admin-links',
            30
        );
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        if ($hook !== 'toplevel_page_url-redirect-checker') {
            return;
        }
        
        wp_enqueue_style(
            'url-redirect-checker-admin',
            URL_REDIRECT_CHECKER_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            URL_REDIRECT_CHECKER_VERSION
        );
        
        wp_enqueue_script(
            'url-redirect-checker-admin',
            URL_REDIRECT_CHECKER_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            URL_REDIRECT_CHECKER_VERSION,
            true
        );
        
        wp_localize_script('url-redirect-checker-admin', 'urcAjax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('urc_nonce')
        ));
    }
    
    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        global $post;
        
        if (is_a($post, 'WP_Post') && !has_shortcode($post->post_content, 'redirect_checker')) {
            return;
        }
        
        wp_enqueue_style(
            'url-redirect-checker-frontend',
            URL_REDIRECT_CHECKER_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            URL_REDIRECT_CHECKER_VERSION
        );
        
        wp_enqueue_script(
            'url-redirect-checker-frontend',
            URL_REDIRECT_CHECKER_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            URL_REDIRECT_CHECKER_VERSION,
            true
        );
        
        wp_localize_script('url-redirect-checker-frontend', 'foxcoRedirectChecker', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('urc_nonce')
        ));
    }
    
    /**
     * Render admin page
     */
    public function render_admin_page() {
        ?>
        <div class="wrap urc-wrapper">
            <h1>
                <?php echo esc_html(get_admin_page_title()); ?>
                <span class="foxco-badge">Powered by Foxco.net</span>
            </h1>
            
            <div class="urc-container">
                <div class="urc-input-section">
                    <div class="urc-card">
                        <h2>Check URL Redirects</h2>
                        <p>Enter URLs (one per line) or upload a CSV file to check where they redirect.</p>
                        
                        <div class="urc-tabs">
                            <button class="urc-tab-button active" data-tab="paste">Paste URLs</button>
                            <button class="urc-tab-button" data-tab="upload">Upload CSV</button>
                        </div>
                        
                        <!-- Paste URLs Tab -->
                        <div id="urc-tab-paste" class="urc-tab-content active">
                            <div class="urc-form-group">
                                <label for="urc-url-input">Enter URLs (one per line):</label>
                                <textarea 
                                    id="urc-url-input" 
                                    class="urc-textarea" 
                                    rows="15" 
                                    placeholder="https://example.com/page1&#10;https://example.com/page2&#10;https://example.com/page3"
                                ></textarea>
                            </div>
                            <button id="urc-check-button" class="button button-primary button-large">
                                Check Redirects
                            </button>
                        </div>
                        
                        <!-- Upload CSV Tab -->
                        <div id="urc-tab-upload" class="urc-tab-content">
                            <div class="urc-form-group">
                                <label for="urc-csv-upload">Upload CSV file:</label>
                                <input type="file" id="urc-csv-upload" accept=".csv" class="urc-file-input">
                                <p class="description">CSV should contain URLs in the first column</p>
                            </div>
                            <button id="urc-upload-button" class="button button-primary button-large">
                                Process CSV
                            </button>
                        </div>
                        
                        <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd;">
                            <h3>Frontend Usage</h3>
                            <p>Add this shortcode to any page or post to display the redirect checker on the frontend:</p>
                            <code style="background: #f0f0f1; padding: 8px 12px; display: inline-block; border-radius: 3px; font-size: 14px;">[redirect_checker]</code>
                            <p style="margin-top: 10px; color: #646970;">The frontend version includes Foxco.net branding and is fully responsive.</p>
                        </div>
                    </div>
                </div>
                
                <div class="urc-results-section">
                    <div class="urc-card">
                        <div class="urc-results-header">
                            <h2>Results</h2>
                            <button id="urc-download-csv" class="button" style="display: none;">
                                Download CSV
                            </button>
                        </div>
                        
                        <div id="urc-progress" class="urc-progress" style="display: none;">
                            <div class="urc-progress-bar">
                                <div class="urc-progress-fill"></div>
                            </div>
                            <p class="urc-progress-text">Checking redirects... <span id="urc-progress-count">0/0</span></p>
                        </div>
                        
                        <div id="urc-results" class="urc-results">
                            <p class="urc-empty-state">Enter URLs or upload a CSV file to get started.</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render shortcode
     */
    public function render_shortcode($atts) {
        $atts = shortcode_atts(array(
            'title' => 'URL Redirect Checker',
            'theme' => 'light' // light or dark
        ), $atts);
        
        ob_start();
        ?>
        <div class="foxco-redirect-checker-container" data-theme="<?php echo esc_attr($atts['theme']); ?>">
            <div class="foxco-header">
                <h2 class="foxco-title"><?php echo esc_html($atts['title']); ?></h2>
                <a href="https://foxco.net" target="_blank" class="foxco-brand">
                    <span class="foxco-brand-text">Powered by</span>
                    <span class="foxco-brand-name">Foxco.net</span>
                </a>
            </div>
            
            <div class="foxco-main">
                <div class="foxco-input-section">
                    <div class="foxco-tabs">
                        <button class="foxco-tab-button active" data-tab="paste">
                            <span class="foxco-tab-icon">📝</span>
                            Paste URLs
                        </button>
                        <button class="foxco-tab-button" data-tab="upload">
                            <span class="foxco-tab-icon">📤</span>
                            Upload CSV
                        </button>
                    </div>
                    
                    <!-- Paste URLs Tab -->
                    <div id="foxco-tab-paste" class="foxco-tab-content active">
                        <p class="foxco-description">Enter URLs (one per line) to check where they redirect:</p>
                        <textarea 
                            id="foxco-url-input" 
                            class="foxco-textarea" 
                            rows="12" 
                            placeholder="https://example.com/page1&#10;https://example.com/page2&#10;https://example.com/page3"
                        ></textarea>
                        <button id="foxco-check-button" class="foxco-button foxco-button-primary">
                            <span class="foxco-button-icon">🔍</span>
                            Check Redirects
                        </button>
                    </div>
                    
                    <!-- Upload CSV Tab -->
                    <div id="foxco-tab-upload" class="foxco-tab-content">
                        <p class="foxco-description">Upload a CSV file with URLs in the first column:</p>
                        <div class="foxco-file-upload-wrapper">
                            <input type="file" id="foxco-csv-upload" accept=".csv" class="foxco-file-input">
                            <label for="foxco-csv-upload" class="foxco-file-label">
                                <span class="foxco-file-icon">📁</span>
                                <span class="foxco-file-text">Choose CSV File</span>
                            </label>
                            <span class="foxco-file-name"></span>
                        </div>
                        <button id="foxco-upload-button" class="foxco-button foxco-button-primary">
                            <span class="foxco-button-icon">⚡</span>
                            Process CSV
                        </button>
                    </div>
                </div>
                
                <div class="foxco-results-section">
                    <div class="foxco-results-header">
                        <h3>Results</h3>
                        <button id="foxco-download-csv" class="foxco-button foxco-button-secondary" style="display: none;">
                            <span class="foxco-button-icon">💾</span>
                            Download CSV
                        </button>
                    </div>
                    
                    <div id="foxco-progress" class="foxco-progress" style="display: none;">
                        <div class="foxco-progress-bar">
                            <div class="foxco-progress-fill"></div>
                        </div>
                        <p class="foxco-progress-text">Checking redirects... <span id="foxco-progress-count">0/0</span></p>
                    </div>
                    
                    <div id="foxco-results" class="foxco-results">
                        <div class="foxco-empty-state">
                            <span class="foxco-empty-icon">🔗</span>
                            <p>Enter URLs or upload a CSV file to get started.</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * AJAX handler for checking redirects
     */
    public function ajax_check_redirects() {
        check_ajax_referer('urc_nonce', 'nonce');
        
        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        
        if (empty($url)) {
            wp_send_json_error('Invalid URL');
            return;
        }
        
        $result = $this->check_single_redirect($url);
        wp_send_json_success($result);
    }
    
    /**
     * AJAX handler for processing CSV
     */
    public function ajax_process_csv() {
        check_ajax_referer('urc_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        if (!isset($_FILES['file'])) {
            wp_send_json_error('No file uploaded');
            return;
        }
        
        $file = $_FILES['file'];
        
        if ($file['error'] !== UPLOAD_ERR_OK) {
            wp_send_json_error('Upload error');
            return;
        }
        
        // Read CSV file
        $urls = array();
        if (($handle = fopen($file['tmp_name'], 'r')) !== false) {
            while (($data = fgetcsv($handle)) !== false) {
                if (!empty($data[0]) && filter_var($data[0], FILTER_VALIDATE_URL)) {
                    $urls[] = $data[0];
                }
            }
            fclose($handle);
        }
        
        wp_send_json_success(array('urls' => $urls));
    }
    
    /**
     * Check single URL redirect
     */
    private function check_single_redirect($url) {
        $result = array(
            'original_url' => $url,
            'final_url' => $url,
            'status_code' => '',
            'redirected' => false,
            'error' => ''
        );
        
        $args = array(
            'timeout' => 10,
            'redirection' => 10,
            'sslverify' => false,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
            )
        );
        
        $response = wp_remote_get($url, $args);
        
        if (is_wp_error($response)) {
            $result['error'] = $response->get_error_message();
            return $result;
        }
        
        // Get final URL after redirects
        $final_url = wp_remote_retrieve_header($response, 'location');
        if (empty($final_url)) {
            // Check if there were redirects in the response
            $response_url = wp_remote_retrieve_response_message($response);
            $cookies = wp_remote_retrieve_cookies($response);
            
            // Try to get the actual final URL
            if (isset($response['http_response'])) {
                $http_response = $response['http_response'];
                if (method_exists($http_response, 'get_response_object')) {
                    $response_obj = $http_response->get_response_object();
                    if (isset($response_obj->url)) {
                        $final_url = $response_obj->url;
                    }
                }
            }
        }
        
        // If we still don't have a final URL, use the original
        if (empty($final_url)) {
            $final_url = $url;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        
        $result['final_url'] = $final_url;
        $result['status_code'] = $status_code;
        $result['redirected'] = ($final_url !== $url);
        
        return $result;
    }
}

// Initialize plugin
new URL_Redirect_Checker();
