/**
 * Foxco URL Redirect Checker - Frontend JavaScript
 * Powered by Foxco.net
 */

(function($) {
    'use strict';
    
    let resultsData = [];
    
    $(document).ready(function() {
        initTabs();
        initFileInput();
        initCheckButton();
        initUploadButton();
        initDownloadButton();
    });
    
    /**
     * Initialize tab switching
     */
    function initTabs() {
        $('.foxco-tab-button').on('click', function() {
            const tabName = $(this).data('tab');
            const container = $(this).closest('.foxco-redirect-checker-container');
            
            // Update active states
            container.find('.foxco-tab-button').removeClass('active');
            $(this).addClass('active');
            
            container.find('.foxco-tab-content').removeClass('active');
            container.find('#foxco-tab-' + tabName).addClass('active');
        });
    }
    
    /**
     * Initialize file input display
     */
    function initFileInput() {
        $('#foxco-csv-upload').on('change', function() {
            const fileName = this.files[0] ? this.files[0].name : '';
            const container = $(this).closest('.foxco-redirect-checker-container');
            if (fileName) {
                container.find('.foxco-file-name').text('Selected: ' + fileName);
            } else {
                container.find('.foxco-file-name').text('');
            }
        });
    }
    
    /**
     * Initialize check button
     */
    function initCheckButton() {
        $(document).on('click', '#foxco-check-button', function() {
            const container = $(this).closest('.foxco-redirect-checker-container');
            const urlInput = container.find('#foxco-url-input').val().trim();
            
            if (!urlInput) {
                alert('Please enter at least one URL.');
                return;
            }
            
            // Split URLs by new line and filter empty lines
            const urls = urlInput.split('\n')
                .map(url => url.trim())
                .filter(url => url.length > 0);
            
            if (urls.length === 0) {
                alert('Please enter valid URLs.');
                return;
            }
            
            processUrls(urls, container);
        });
    }
    
    /**
     * Initialize upload button
     */
    function initUploadButton() {
        $(document).on('click', '#foxco-upload-button', function() {
            const container = $(this).closest('.foxco-redirect-checker-container');
            const fileInput = container.find('#foxco-csv-upload')[0];
            
            if (!fileInput.files || fileInput.files.length === 0) {
                alert('Please select a CSV file.');
                return;
            }
            
            const file = fileInput.files[0];
            const formData = new FormData();
            formData.append('action', 'urc_process_csv');
            formData.append('nonce', foxcoRedirectChecker.nonce);
            formData.append('file', file);
            
            $.ajax({
                url: foxcoRedirectChecker.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success && response.data.urls) {
                        processUrls(response.data.urls, container);
                    } else {
                        alert('Error processing CSV file.');
                    }
                },
                error: function() {
                    alert('Error uploading file.');
                }
            });
        });
    }
    
    /**
     * Initialize download button
     */
    function initDownloadButton() {
        $(document).on('click', '#foxco-download-csv', function() {
            downloadCSV();
        });
    }
    
    /**
     * Process URLs and check redirects
     */
    function processUrls(urls, container) {
        resultsData = [];
        let processed = 0;
        const total = urls.length;
        
        // Show progress bar
        container.find('#foxco-progress').show();
        container.find('#foxco-results').html('<div class="foxco-empty-state"><span class="foxco-empty-icon">⏳</span><p>Processing URLs...</p></div>');
        updateProgress(0, total, container);
        
        // Disable buttons
        container.find('#foxco-check-button, #foxco-upload-button').prop('disabled', true).css('opacity', '0.6');
        
        // Process each URL
        const promises = urls.map(url => {
            return checkRedirect(url).then(result => {
                processed++;
                resultsData.push(result);
                updateProgress(processed, total, container);
            });
        });
        
        // Wait for all to complete
        Promise.all(promises).then(() => {
            displayResults(container);
            
            // Hide progress bar
            container.find('#foxco-progress').hide();
            
            // Enable buttons
            container.find('#foxco-check-button, #foxco-upload-button').prop('disabled', false).css('opacity', '1');
            
            // Show download button
            container.find('#foxco-download-csv').show();
        });
    }
    
    /**
     * Check single URL redirect via AJAX
     */
    function checkRedirect(url) {
        return new Promise((resolve) => {
            $.ajax({
                url: foxcoRedirectChecker.ajax_url,
                type: 'POST',
                data: {
                    action: 'urc_check_redirects',
                    nonce: foxcoRedirectChecker.nonce,
                    url: url
                },
                success: function(response) {
                    if (response.success) {
                        resolve(response.data);
                    } else {
                        resolve({
                            original_url: url,
                            final_url: url,
                            status_code: '',
                            redirected: false,
                            error: 'Failed to check URL'
                        });
                    }
                },
                error: function() {
                    resolve({
                        original_url: url,
                        final_url: url,
                        status_code: '',
                        redirected: false,
                        error: 'Request failed'
                    });
                }
            });
        });
    }
    
    /**
     * Update progress bar
     */
    function updateProgress(current, total, container) {
        const percentage = Math.round((current / total) * 100);
        container.find('.foxco-progress-fill').css('width', percentage + '%');
        container.find('#foxco-progress-count').text(current + '/' + total);
    }
    
    /**
     * Display results in table
     */
    function displayResults(container) {
        if (resultsData.length === 0) {
            container.find('#foxco-results').html('<div class="foxco-empty-state"><span class="foxco-empty-icon">❌</span><p>No results to display.</p></div>');
            return;
        }
        
        // Calculate summary stats
        const totalUrls = resultsData.length;
        const redirectCount = resultsData.filter(r => r.redirected).length;
        const errorCount = resultsData.filter(r => r.error).length;
        const noRedirectCount = totalUrls - redirectCount - errorCount;
        
        let html = '<div class="foxco-summary">';
        html += '<div class="foxco-stat"><span class="foxco-stat-value">' + totalUrls + '</span><span class="foxco-stat-label">Total URLs</span></div>';
        html += '<div class="foxco-stat"><span class="foxco-stat-value">' + redirectCount + '</span><span class="foxco-stat-label">Redirects</span></div>';
        html += '<div class="foxco-stat"><span class="foxco-stat-value">' + noRedirectCount + '</span><span class="foxco-stat-label">No Redirect</span></div>';
        html += '<div class="foxco-stat"><span class="foxco-stat-value">' + errorCount + '</span><span class="foxco-stat-label">Errors</span></div>';
        html += '</div>';
        
        html += '<table class="foxco-results-table">';
        html += '<thead><tr>';
        html += '<th style="width: 40%;">A: Original URL</th>';
        html += '<th style="width: 40%;">B: Final Destination URL</th>';
        html += '<th style="width: 10%;">Status</th>';
        html += '<th style="width: 10%;">Code</th>';
        html += '</tr></thead>';
        html += '<tbody>';
        
        resultsData.forEach(function(result) {
            let statusBadge = '';
            let statusClass = '';
            
            if (result.error) {
                statusBadge = '<span class="foxco-status-badge error">Error</span>';
                statusClass = 'error';
            } else if (result.redirected) {
                statusBadge = '<span class="foxco-status-badge redirect">Redirect</span>';
                statusClass = 'redirect';
            } else {
                statusBadge = '<span class="foxco-status-badge no-redirect">No Redirect</span>';
                statusClass = 'no-redirect';
            }
            
            html += '<tr class="foxco-row-' + statusClass + '">';
            html += '<td><div class="foxco-url">' + escapeHtml(result.original_url) + '</div></td>';
            html += '<td><div class="foxco-url">' + escapeHtml(result.final_url) + '</div></td>';
            html += '<td>' + statusBadge + '</td>';
            html += '<td><span class="foxco-status-code">' + (result.status_code || '-') + '</span></td>';
            html += '</tr>';
            
            if (result.error) {
                html += '<tr class="foxco-row-error"><td colspan="4" style="padding-left: 40px; color: #f76b1c; font-size: 12px;">Error: ' + escapeHtml(result.error) + '</td></tr>';
            }
        });
        
        html += '</tbody></table>';
        
        container.find('#foxco-results').html(html);
    }
    
    /**
     * Download results as CSV
     */
    function downloadCSV() {
        if (resultsData.length === 0) {
            alert('No data to download.');
            return;
        }
        
        let csv = 'A: Original URL,B: Final Destination URL,Status,Status Code,Error\n';
        
        resultsData.forEach(function(result) {
            let status = result.error ? 'Error' : (result.redirected ? 'Redirect' : 'No Redirect');
            let row = [
                '"' + result.original_url.replace(/"/g, '""') + '"',
                '"' + result.final_url.replace(/"/g, '""') + '"',
                status,
                result.status_code || '',
                result.error ? '"' + result.error.replace(/"/g, '""') + '"' : ''
            ];
            csv += row.join(',') + '\n';
        });
        
        // Create download link
        const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        
        link.setAttribute('href', url);
        link.setAttribute('download', 'foxco-redirect-results-' + Date.now() + '.csv');
        link.style.visibility = 'hidden';
        
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }
    
    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
    
})(jQuery);
