/**
 * URL Redirect Checker - Admin JavaScript
 */

(function($) {
    'use strict';
    
    let resultsData = [];
    
    $(document).ready(function() {
        initTabs();
        initCheckButton();
        initUploadButton();
        initDownloadButton();
    });
    
    /**
     * Initialize tab switching
     */
    function initTabs() {
        $('.urc-tab-button').on('click', function() {
            const tabName = $(this).data('tab');
            
            // Update active states
            $('.urc-tab-button').removeClass('active');
            $(this).addClass('active');
            
            $('.urc-tab-content').removeClass('active');
            $('#urc-tab-' + tabName).addClass('active');
        });
    }
    
    /**
     * Initialize check button
     */
    function initCheckButton() {
        $('#urc-check-button').on('click', function() {
            const urlInput = $('#urc-url-input').val().trim();
            
            if (!urlInput) {
                alert('Please enter at least one URL.');
                return;
            }
            
            // Split URLs by new line and filter empty lines
            const urls = urlInput.split('\n')
                .map(url => url.trim())
                .filter(url => url.length > 0);
            
            if (urls.length === 0) {
                alert('Please enter valid URLs.');
                return;
            }
            
            processUrls(urls);
        });
    }
    
    /**
     * Initialize upload button
     */
    function initUploadButton() {
        $('#urc-upload-button').on('click', function() {
            const fileInput = $('#urc-csv-upload')[0];
            
            if (!fileInput.files || fileInput.files.length === 0) {
                alert('Please select a CSV file.');
                return;
            }
            
            const file = fileInput.files[0];
            const formData = new FormData();
            formData.append('action', 'urc_process_csv');
            formData.append('nonce', urcAjax.nonce);
            formData.append('file', file);
            
            $.ajax({
                url: urcAjax.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success && response.data.urls) {
                        processUrls(response.data.urls);
                    } else {
                        alert('Error processing CSV file.');
                    }
                },
                error: function() {
                    alert('Error uploading file.');
                }
            });
        });
    }
    
    /**
     * Initialize download button
     */
    function initDownloadButton() {
        $('#urc-download-csv').on('click', function() {
            downloadCSV();
        });
    }
    
    /**
     * Process URLs and check redirects
     */
    function processUrls(urls) {
        resultsData = [];
        let processed = 0;
        const total = urls.length;
        
        // Show progress bar
        $('#urc-progress').show();
        $('#urc-results').html('<p class="urc-empty-state">Processing URLs...</p>');
        updateProgress(0, total);
        
        // Disable buttons
        $('#urc-check-button, #urc-upload-button').prop('disabled', true).addClass('disabled');
        
        // Process each URL
        const promises = urls.map(url => {
            return checkRedirect(url).then(result => {
                processed++;
                resultsData.push(result);
                updateProgress(processed, total);
            });
        });
        
        // Wait for all to complete
        Promise.all(promises).then(() => {
            displayResults();
            
            // Hide progress bar
            $('#urc-progress').hide();
            
            // Enable buttons
            $('#urc-check-button, #urc-upload-button').prop('disabled', false).removeClass('disabled');
            
            // Show download button
            $('#urc-download-csv').show();
        });
    }
    
    /**
     * Check single URL redirect via AJAX
     */
    function checkRedirect(url) {
        return new Promise((resolve) => {
            $.ajax({
                url: urcAjax.ajax_url,
                type: 'POST',
                data: {
                    action: 'urc_check_redirects',
                    nonce: urcAjax.nonce,
                    url: url
                },
                success: function(response) {
                    if (response.success) {
                        resolve(response.data);
                    } else {
                        resolve({
                            original_url: url,
                            final_url: url,
                            status_code: '',
                            redirected: false,
                            error: 'Failed to check URL'
                        });
                    }
                },
                error: function() {
                    resolve({
                        original_url: url,
                        final_url: url,
                        status_code: '',
                        redirected: false,
                        error: 'Request failed'
                    });
                }
            });
        });
    }
    
    /**
     * Update progress bar
     */
    function updateProgress(current, total) {
        const percentage = Math.round((current / total) * 100);
        $('.urc-progress-fill').css('width', percentage + '%');
        $('#urc-progress-count').text(current + '/' + total);
    }
    
    /**
     * Display results in table
     */
    function displayResults() {
        if (resultsData.length === 0) {
            $('#urc-results').html('<p class="urc-empty-state">No results to display.</p>');
            return;
        }
        
        // Calculate summary stats
        const totalUrls = resultsData.length;
        const redirectCount = resultsData.filter(r => r.redirected).length;
        const errorCount = resultsData.filter(r => r.error).length;
        const noRedirectCount = totalUrls - redirectCount - errorCount;
        
        let html = '<div class="urc-summary">';
        html += '<div class="urc-stat"><span class="urc-stat-value">' + totalUrls + '</span><span class="urc-stat-label">Total URLs</span></div>';
        html += '<div class="urc-stat"><span class="urc-stat-value">' + redirectCount + '</span><span class="urc-stat-label">Redirects</span></div>';
        html += '<div class="urc-stat"><span class="urc-stat-value">' + noRedirectCount + '</span><span class="urc-stat-label">No Redirect</span></div>';
        html += '<div class="urc-stat"><span class="urc-stat-value">' + errorCount + '</span><span class="urc-stat-label">Errors</span></div>';
        html += '</div>';
        
        html += '<table class="urc-results-table">';
        html += '<thead><tr>';
        html += '<th style="width: 40%;">A: Original URL</th>';
        html += '<th style="width: 40%;">B: Final Destination URL</th>';
        html += '<th style="width: 10%;">Status</th>';
        html += '<th style="width: 10%;">Code</th>';
        html += '</tr></thead>';
        html += '<tbody>';
        
        resultsData.forEach(function(result) {
            let statusBadge = '';
            let statusClass = '';
            
            if (result.error) {
                statusBadge = '<span class="urc-status-badge error">Error</span>';
                statusClass = 'error';
            } else if (result.redirected) {
                statusBadge = '<span class="urc-status-badge redirect">Redirect</span>';
                statusClass = 'redirect';
            } else {
                statusBadge = '<span class="urc-status-badge no-redirect">No Redirect</span>';
                statusClass = 'no-redirect';
            }
            
            html += '<tr class="urc-row-' + statusClass + '">';
            html += '<td><div class="urc-url">' + escapeHtml(result.original_url) + '</div></td>';
            html += '<td><div class="urc-url">' + escapeHtml(result.final_url) + '</div></td>';
            html += '<td>' + statusBadge + '</td>';
            html += '<td><span class="urc-status-code">' + (result.status_code || '-') + '</span></td>';
            html += '</tr>';
            
            if (result.error) {
                html += '<tr class="urc-row-error"><td colspan="4" style="padding-left: 40px; color: #d63638; font-size: 12px;">Error: ' + escapeHtml(result.error) + '</td></tr>';
            }
        });
        
        html += '</tbody></table>';
        
        $('#urc-results').html(html);
    }
    
    /**
     * Download results as CSV
     */
    function downloadCSV() {
        if (resultsData.length === 0) {
            alert('No data to download.');
            return;
        }
        
        let csv = 'A: Original URL,B: Final Destination URL,Status,Status Code,Error\n';
        
        resultsData.forEach(function(result) {
            let status = result.error ? 'Error' : (result.redirected ? 'Redirect' : 'No Redirect');
            let row = [
                '"' + result.original_url.replace(/"/g, '""') + '"',
                '"' + result.final_url.replace(/"/g, '""') + '"',
                status,
                result.status_code || '',
                result.error ? '"' + result.error.replace(/"/g, '""') + '"' : ''
            ];
            csv += row.join(',') + '\n';
        });
        
        // Create download link
        const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        
        link.setAttribute('href', url);
        link.setAttribute('download', 'redirect-results-' + Date.now() + '.csv');
        link.style.visibility = 'hidden';
        
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }
    
    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
    
})(jQuery);
