<?php
/**
 * Plugin Name: Foxco LiteSpeed/QUIC Cleanup
 * Plugin URI: https://foxco.net
 * Description: Removes all traces of LiteSpeed Cache and QUIC services from WordPress database, configuration, and files
 * Version: 1.0.0
 * Author: Foxco.net
 * Author URI: https://foxco.net
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: foxco-litespeed-cleanup
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Foxco_LiteSpeed_Cleanup {
    
    private $log = array();
    private $items_cleaned = 0;
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_foxco_run_litespeed_cleanup', array($this, 'ajax_run_cleanup'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_management_page(
            'LiteSpeed/QUIC Cleanup',
            'LiteSpeed Cleanup',
            'manage_options',
            'foxco-litespeed-cleanup',
            array($this, 'admin_page')
        );
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'tools_page_foxco-litespeed-cleanup') {
            return;
        }
        
        wp_enqueue_style(
            'foxco-litespeed-cleanup',
            plugins_url('assets/admin.css', __FILE__),
            array(),
            '1.0.0'
        );
    }
    
    /**
     * Admin page
     */
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1>
                <span class="dashicons dashicons-admin-tools" style="font-size: 32px; margin-right: 10px;"></span>
                Foxco LiteSpeed/QUIC Cleanup
            </h1>
            
            <div class="card" style="max-width: 800px; margin-top: 20px;">
                <h2>Remove LiteSpeed Cache & QUIC Services</h2>
                <p>This tool will remove all traces of LiteSpeed Cache and QUIC services from your WordPress installation, including:</p>
                <ul style="list-style: disc; margin-left: 20px;">
                    <li>Database options and settings</li>
                    <li>Post meta data</li>
                    <li>User meta data</li>
                    <li>Transients</li>
                    <li>Cron jobs</li>
                    <li>.htaccess rules</li>
                    <li>Cache directories</li>
                    <li>Configuration files</li>
                </ul>
                
                <div class="notice notice-warning inline" style="margin: 20px 0;">
                    <p><strong>Warning:</strong> This action cannot be undone. Please ensure you have a backup before proceeding.</p>
                </div>
                
                <p>
                    <button type="button" id="foxco-run-cleanup" class="button button-primary button-large">
                        <span class="dashicons dashicons-trash" style="margin-top: 3px;"></span>
                        Run Cleanup Now
                    </button>
                    <span id="foxco-cleanup-spinner" class="spinner" style="float: none; margin-left: 10px;"></span>
                </p>
            </div>
            
            <div id="foxco-cleanup-results" style="display: none; margin-top: 20px;">
                <div class="card" style="max-width: 800px;">
                    <h2>Cleanup Results</h2>
                    <div id="foxco-cleanup-log"></div>
                </div>
            </div>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#foxco-run-cleanup').on('click', function() {
                if (!confirm('Are you sure you want to remove all LiteSpeed/QUIC data? This cannot be undone.')) {
                    return;
                }
                
                var $button = $(this);
                var $spinner = $('#foxco-cleanup-spinner');
                var $results = $('#foxco-cleanup-results');
                var $log = $('#foxco-cleanup-log');
                
                $button.prop('disabled', true);
                $spinner.addClass('is-active');
                $results.hide();
                $log.html('');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'foxco_run_litespeed_cleanup',
                        nonce: '<?php echo wp_create_nonce('foxco_litespeed_cleanup'); ?>'
                    },
                    success: function(response) {
                        $spinner.removeClass('is-active');
                        $button.prop('disabled', false);
                        
                        if (response.success) {
                            $log.html('<div class="notice notice-success inline"><p><strong>Cleanup completed successfully!</strong></p></div>');
                            $log.append('<p><strong>Items cleaned: ' + response.data.items_cleaned + '</strong></p>');
                            $log.append('<div style="background: #f0f0f1; padding: 15px; border-radius: 4px; max-height: 400px; overflow-y: auto;"><pre>' + response.data.log.join('\n') + '</pre></div>');
                        } else {
                            $log.html('<div class="notice notice-error inline"><p><strong>Error: </strong>' + response.data.message + '</p></div>');
                        }
                        
                        $results.show();
                    },
                    error: function() {
                        $spinner.removeClass('is-active');
                        $button.prop('disabled', false);
                        $log.html('<div class="notice notice-error inline"><p><strong>Error: </strong>AJAX request failed.</p></div>');
                        $results.show();
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX handler for cleanup
     */
    public function ajax_run_cleanup() {
        check_ajax_referer('foxco_litespeed_cleanup', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }
        
        $this->run_cleanup();
        
        wp_send_json_success(array(
            'items_cleaned' => $this->items_cleaned,
            'log' => $this->log
        ));
    }
    
    /**
     * Run the cleanup process
     */
    private function run_cleanup() {
        global $wpdb;
        
        $this->log[] = "=== Starting LiteSpeed/QUIC Cleanup ===";
        $this->log[] = "Time: " . current_time('mysql');
        $this->log[] = "";
        
        // 1. Clean options table
        $this->clean_options();
        
        // 2. Clean post meta
        $this->clean_post_meta();
        
        // 3. Clean user meta
        $this->clean_user_meta();
        
        // 4. Clean transients
        $this->clean_transients();
        
        // 5. Clean cron jobs
        $this->clean_cron_jobs();
        
        // 6. Clean .htaccess
        $this->clean_htaccess();
        
        // 7. Clean cache directories
        $this->clean_cache_directories();
        
        // 8. Clean configuration files
        $this->clean_config_files();
        
        // 9. Clean wp-config.php constants
        $this->clean_wp_config_constants();
        
        $this->log[] = "";
        $this->log[] = "=== Cleanup Complete ===";
        $this->log[] = "Total items cleaned: " . $this->items_cleaned;
    }
    
    /**
     * Clean options table
     */
    private function clean_options() {
        global $wpdb;
        
        $this->log[] = "--- Cleaning Options Table ---";
        
        $patterns = array(
            'litespeed%',
            'lscache%',
            'lscwp%',
            'quic%',
            '_litespeed%',
            '_lscache%',
            'litespeed_cache%',
            'litespeed-cache%'
        );
        
        foreach ($patterns as $pattern) {
            $results = $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                    $pattern
                )
            );
            
            if ($results > 0) {
                $this->items_cleaned += $results;
                $this->log[] = "Removed {$results} options matching pattern: {$pattern}";
            }
        }
        
        $this->log[] = "";
    }
    
    /**
     * Clean post meta
     */
    private function clean_post_meta() {
        global $wpdb;
        
        $this->log[] = "--- Cleaning Post Meta ---";
        
        $patterns = array(
            'litespeed%',
            'lscache%',
            'lscwp%',
            '_litespeed%',
            '_lscache%'
        );
        
        foreach ($patterns as $pattern) {
            $results = $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE %s",
                    $pattern
                )
            );
            
            if ($results > 0) {
                $this->items_cleaned += $results;
                $this->log[] = "Removed {$results} post meta entries matching pattern: {$pattern}";
            }
        }
        
        $this->log[] = "";
    }
    
    /**
     * Clean user meta
     */
    private function clean_user_meta() {
        global $wpdb;
        
        $this->log[] = "--- Cleaning User Meta ---";
        
        $patterns = array(
            'litespeed%',
            'lscache%',
            'lscwp%',
            '_litespeed%',
            '_lscache%'
        );
        
        foreach ($patterns as $pattern) {
            $results = $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE %s",
                    $pattern
                )
            );
            
            if ($results > 0) {
                $this->items_cleaned += $results;
                $this->log[] = "Removed {$results} user meta entries matching pattern: {$pattern}";
            }
        }
        
        $this->log[] = "";
    }
    
    /**
     * Clean transients
     */
    private function clean_transients() {
        global $wpdb;
        
        $this->log[] = "--- Cleaning Transients ---";
        
        $patterns = array(
            '_transient_litespeed%',
            '_transient_timeout_litespeed%',
            '_transient_lscache%',
            '_transient_timeout_lscache%',
            '_transient_lscwp%',
            '_transient_timeout_lscwp%',
            '_transient_quic%',
            '_transient_timeout_quic%'
        );
        
        foreach ($patterns as $pattern) {
            $results = $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                    $pattern
                )
            );
            
            if ($results > 0) {
                $this->items_cleaned += $results;
                $this->log[] = "Removed {$results} transients matching pattern: {$pattern}";
            }
        }
        
        $this->log[] = "";
    }
    
    /**
     * Clean cron jobs
     */
    private function clean_cron_jobs() {
        $this->log[] = "--- Cleaning Cron Jobs ---";
        
        $crons = _get_cron_array();
        $cleaned = 0;
        
        foreach ($crons as $timestamp => $cron) {
            foreach ($cron as $hook => $events) {
                if (strpos($hook, 'litespeed') !== false || 
                    strpos($hook, 'lscache') !== false || 
                    strpos($hook, 'lscwp') !== false ||
                    strpos($hook, 'quic') !== false) {
                    
                    foreach ($events as $key => $event) {
                        wp_unschedule_event($timestamp, $hook, $event['args']);
                        $cleaned++;
                        $this->log[] = "Removed cron job: {$hook}";
                    }
                }
            }
        }
        
        if ($cleaned > 0) {
            $this->items_cleaned += $cleaned;
            $this->log[] = "Total cron jobs removed: {$cleaned}";
        } else {
            $this->log[] = "No LiteSpeed/QUIC cron jobs found";
        }
        
        $this->log[] = "";
    }
    
    /**
     * Clean .htaccess rules
     */
    private function clean_htaccess() {
        $this->log[] = "--- Cleaning .htaccess Rules ---";
        
        $htaccess_file = ABSPATH . '.htaccess';
        
        if (!file_exists($htaccess_file)) {
            $this->log[] = ".htaccess file not found";
            $this->log[] = "";
            return;
        }
        
        if (!is_writable($htaccess_file)) {
            $this->log[] = "Warning: .htaccess is not writable. Please manually remove LiteSpeed rules.";
            $this->log[] = "";
            return;
        }
        
        $content = file_get_contents($htaccess_file);
        $original_content = $content;
        
        // Remove LiteSpeed Cache blocks
        $content = preg_replace(
            '/# BEGIN LSCACHE.*?# END LSCACHE\s*/s',
            '',
            $content
        );
        
        $content = preg_replace(
            '/# BEGIN LiteSpeed.*?# END LiteSpeed\s*/s',
            '',
            $content
        );
        
        // Remove individual LiteSpeed directives
        $litespeed_directives = array(
            'CacheLookup',
            'RewriteEngine',
            'LiteSpeed',
            'LSPHP',
            'noabort',
            'noconntimeout'
        );
        
        $lines = explode("\n", $content);
        $cleaned_lines = array();
        
        foreach ($lines as $line) {
            $is_litespeed = false;
            foreach ($litespeed_directives as $directive) {
                if (stripos($line, $directive) !== false && 
                    (stripos($line, 'litespeed') !== false || stripos($line, 'lscache') !== false)) {
                    $is_litespeed = true;
                    break;
                }
            }
            
            if (!$is_litespeed) {
                $cleaned_lines[] = $line;
            }
        }
        
        $content = implode("\n", $cleaned_lines);
        
        // Remove multiple consecutive blank lines
        $content = preg_replace("/\n{3,}/", "\n\n", $content);
        
        if ($content !== $original_content) {
            file_put_contents($htaccess_file, $content);
            $this->items_cleaned++;
            $this->log[] = ".htaccess cleaned - LiteSpeed rules removed";
        } else {
            $this->log[] = "No LiteSpeed rules found in .htaccess";
        }
        
        $this->log[] = "";
    }
    
    /**
     * Clean cache directories
     */
    private function clean_cache_directories() {
        $this->log[] = "--- Cleaning Cache Directories ---";
        
        $cache_dirs = array(
            WP_CONTENT_DIR . '/cache/litespeed',
            WP_CONTENT_DIR . '/cache/lscache',
            WP_CONTENT_DIR . '/litespeed',
            WP_CONTENT_DIR . '/lscache',
            ABSPATH . 'lscache'
        );
        
        foreach ($cache_dirs as $dir) {
            if (file_exists($dir)) {
                $this->recursive_delete($dir);
                $this->items_cleaned++;
                $this->log[] = "Removed directory: {$dir}";
            }
        }
        
        $this->log[] = "";
    }
    
    /**
     * Clean configuration files
     */
    private function clean_config_files() {
        $this->log[] = "--- Cleaning Configuration Files ---";
        
        $config_files = array(
            ABSPATH . '.litespeed_conf.dat',
            ABSPATH . 'lscache.conf',
            WP_CONTENT_DIR . '/.litespeed_conf.dat',
            WP_CONTENT_DIR . '/lscache.conf'
        );
        
        foreach ($config_files as $file) {
            if (file_exists($file)) {
                unlink($file);
                $this->items_cleaned++;
                $this->log[] = "Removed configuration file: {$file}";
            }
        }
        
        $this->log[] = "";
    }
    
    /**
     * Clean wp-config.php constants
     */
    private function clean_wp_config_constants() {
        $this->log[] = "--- Checking wp-config.php Constants ---";
        
        $wp_config = ABSPATH . 'wp-config.php';
        
        if (!file_exists($wp_config)) {
            $this->log[] = "wp-config.php not found";
            $this->log[] = "";
            return;
        }
        
        if (!is_writable($wp_config)) {
            $this->log[] = "Warning: wp-config.php is not writable.";
            $this->log[] = "Please manually remove any LSCACHE or LITESPEED constants.";
            $this->log[] = "";
            return;
        }
        
        $content = file_get_contents($wp_config);
        $original_content = $content;
        
        // Remove LiteSpeed-related constants
        $patterns = array(
            "/define\s*\(\s*['\"]LSCACHE_.*?['\"].*?\);?\s*/i",
            "/define\s*\(\s*['\"]LITESPEED_.*?['\"].*?\);?\s*/i",
            "/define\s*\(\s*['\"]LSCWP_.*?['\"].*?\);?\s*/i",
        );
        
        foreach ($patterns as $pattern) {
            $content = preg_replace($pattern, '', $content);
        }
        
        // Remove comments related to LiteSpeed
        $content = preg_replace("/\/\/.*?(litespeed|lscache|lscwp).*?\n/i", '', $content);
        $content = preg_replace("/\/\*.*?(litespeed|lscache|lscwp).*?\*\/\s*/is", '', $content);
        
        if ($content !== $original_content) {
            file_put_contents($wp_config, $content);
            $this->items_cleaned++;
            $this->log[] = "Removed LiteSpeed constants from wp-config.php";
        } else {
            $this->log[] = "No LiteSpeed constants found in wp-config.php";
        }
        
        $this->log[] = "";
    }
    
    /**
     * Recursively delete directory
     */
    private function recursive_delete($dir) {
        if (!file_exists($dir)) {
            return;
        }
        
        if (is_file($dir)) {
            unlink($dir);
            return;
        }
        
        $files = array_diff(scandir($dir), array('.', '..'));
        
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->recursive_delete($path) : unlink($path);
        }
        
        rmdir($dir);
    }
}

// Initialize the plugin
new Foxco_LiteSpeed_Cleanup();
