<?php
/**
 * Plugin Name: Foxco Debug Log Viewer
 * Plugin URI: https://foxco.net
 * Description: Simple debug.log viewer for WordPress with real-time monitoring and log management
 * Version: 1.0.0
 * Author: Foxco
 * Author URI: https://foxco.net
 * License: GPL v2 or later
 * Text Domain: foxco-debug-viewer
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Foxco_Debug_Log_Viewer {
    
    private $log_file;
    
    public function __construct() {
        $this->log_file = WP_CONTENT_DIR . '/debug.log';
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // AJAX handlers
        add_action('wp_ajax_foxco_get_log', array($this, 'ajax_get_log'));
        add_action('wp_ajax_foxco_clear_log', array($this, 'ajax_clear_log'));
        add_action('wp_ajax_foxco_download_log', array($this, 'ajax_download_log'));
        
        // Admin scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    public function add_admin_menu() {
        add_management_page(
            'Debug Log Viewer',
            'Debug Log',
            'manage_options',
            'foxco-debug-log',
            array($this, 'render_admin_page')
        );
    }
    
    public function enqueue_scripts($hook) {
        if ($hook !== 'tools_page_foxco-debug-log') {
            return;
        }
        
        wp_enqueue_style(
            'foxco-debug-viewer',
            plugin_dir_url(__FILE__) . 'assets/style.css',
            array(),
            '1.0.0'
        );
        
        wp_enqueue_script(
            'foxco-debug-viewer',
            plugin_dir_url(__FILE__) . 'assets/script.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        wp_localize_script('foxco-debug-viewer', 'foxcoDebugViewer', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('foxco_debug_viewer')
        ));
    }
    
    public function render_admin_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }
        
        $log_exists = file_exists($this->log_file);
        $log_size = $log_exists ? size_format(filesize($this->log_file), 2) : '0 B';
        $wp_debug_enabled = defined('WP_DEBUG') && WP_DEBUG;
        $wp_debug_log_enabled = defined('WP_DEBUG_LOG') && WP_DEBUG_LOG;
        
        ?>
        <div class="wrap foxco-debug-viewer">
            <h1>
                <span class="dashicons dashicons-admin-tools"></span>
                Foxco Debug Log Viewer
            </h1>
            
            <div class="foxco-debug-header">
                <div class="foxco-debug-info">
                    <div class="info-item">
                        <strong>Log File:</strong> 
                        <?php echo $log_exists ? '<span class="status-active">Active</span>' : '<span class="status-inactive">Not Found</span>'; ?>
                    </div>
                    <div class="info-item">
                        <strong>File Size:</strong> <?php echo esc_html($log_size); ?>
                    </div>
                    <div class="info-item">
                        <strong>WP_DEBUG:</strong> 
                        <?php echo $wp_debug_enabled ? '<span class="status-active">Enabled</span>' : '<span class="status-inactive">Disabled</span>'; ?>
                    </div>
                    <div class="info-item">
                        <strong>WP_DEBUG_LOG:</strong> 
                        <?php echo $wp_debug_log_enabled ? '<span class="status-active">Enabled</span>' : '<span class="status-inactive">Disabled</span>'; ?>
                    </div>
                </div>
                
                <div class="foxco-debug-actions">
                    <button id="foxco-refresh-log" class="button button-primary">
                        <span class="dashicons dashicons-update"></span> Refresh
                    </button>
                    <button id="foxco-auto-refresh" class="button" data-active="0">
                        <span class="dashicons dashicons-backup"></span> Auto-Refresh: OFF
                    </button>
                    <button id="foxco-download-log" class="button">
                        <span class="dashicons dashicons-download"></span> Download
                    </button>
                    <button id="foxco-clear-log" class="button button-secondary">
                        <span class="dashicons dashicons-trash"></span> Clear Log
                    </button>
                </div>
            </div>
            
            <?php if (!$wp_debug_enabled || !$wp_debug_log_enabled): ?>
            <div class="notice notice-warning">
                <p>
                    <strong>Debug logging is not fully enabled.</strong> 
                    To enable debug logging, add these lines to your wp-config.php file:
                </p>
                <pre>define('WP_DEBUG', true);
define('WP_DEBUG_LOG', true);
define('WP_DEBUG_DISPLAY', false);</pre>
            </div>
            <?php endif; ?>
            
            <div class="foxco-debug-controls">
                <label>
                    <input type="checkbox" id="foxco-filter-errors" checked> 
                    Show Errors
                </label>
                <label>
                    <input type="checkbox" id="foxco-filter-warnings" checked> 
                    Show Warnings
                </label>
                <label>
                    <input type="checkbox" id="foxco-filter-notices" checked> 
                    Show Notices
                </label>
                <label>
                    <input type="number" id="foxco-line-limit" min="50" max="10000" step="50" value="500">
                    Lines to Display
                </label>
                <input type="text" id="foxco-search" placeholder="Search log..." />
            </div>
            
            <div id="foxco-log-container" class="foxco-log-container">
                <div class="foxco-loading">Loading debug log...</div>
            </div>
        </div>
        <?php
    }
    
    public function ajax_get_log() {
        check_ajax_referer('foxco_debug_viewer', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $lines = isset($_POST['lines']) ? intval($_POST['lines']) : 500;
        
        if (!file_exists($this->log_file)) {
            wp_send_json_success(array(
                'content' => 'No debug log found. Debug logging may not be enabled.',
                'size' => 0,
                'modified' => ''
            ));
        }
        
        $content = $this->get_log_tail($lines);
        
        wp_send_json_success(array(
            'content' => $content,
            'size' => size_format(filesize($this->log_file), 2),
            'modified' => date('Y-m-d H:i:s', filemtime($this->log_file))
        ));
    }
    
    public function ajax_clear_log() {
        check_ajax_referer('foxco_debug_viewer', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        if (file_exists($this->log_file)) {
            file_put_contents($this->log_file, '');
            wp_send_json_success('Log file cleared successfully');
        } else {
            wp_send_json_error('Log file not found');
        }
    }
    
    private function get_log_tail($lines = 500) {
        if (!file_exists($this->log_file)) {
            return '';
        }
        
        $file = new SplFileObject($this->log_file, 'r');
        $file->seek(PHP_INT_MAX);
        $total_lines = $file->key() + 1;
        
        $start_line = max(0, $total_lines - $lines);
        
        $content = array();
        $file->seek($start_line);
        
        while (!$file->eof()) {
            $line = $file->current();
            if ($line !== false) {
                $content[] = $line;
            }
            $file->next();
        }
        
        return implode('', $content);
    }
}

// Initialize plugin
new Foxco_Debug_Log_Viewer();
