jQuery(document).ready(function($) {
    let autoRefreshInterval = null;
    let currentSearchTerm = '';
    
    // Load log on page load
    loadLog();
    
    // Refresh button
    $('#foxco-refresh-log').on('click', function() {
        loadLog();
    });
    
    // Auto-refresh toggle
    $('#foxco-auto-refresh').on('click', function() {
        const $btn = $(this);
        const isActive = $btn.data('active');
        
        if (isActive) {
            clearInterval(autoRefreshInterval);
            $btn.data('active', 0);
            $btn.removeClass('active');
            $btn.html('<span class="dashicons dashicons-backup"></span> Auto-Refresh: OFF');
        } else {
            autoRefreshInterval = setInterval(loadLog, 5000);
            $btn.data('active', 1);
            $btn.addClass('active');
            $btn.html('<span class="dashicons dashicons-backup"></span> Auto-Refresh: ON (5s)');
        }
    });
    
    // Clear log
    $('#foxco-clear-log').on('click', function() {
        if (!confirm('Are you sure you want to clear the debug log? This cannot be undone.')) {
            return;
        }
        
        $.ajax({
            url: foxcoDebugViewer.ajax_url,
            type: 'POST',
            data: {
                action: 'foxco_clear_log',
                nonce: foxcoDebugViewer.nonce
            },
            success: function(response) {
                if (response.success) {
                    alert('Debug log cleared successfully');
                    loadLog();
                } else {
                    alert('Error: ' + response.data);
                }
            },
            error: function() {
                alert('An error occurred while clearing the log');
            }
        });
    });
    
    // Download log
    $('#foxco-download-log').on('click', function() {
        const lines = $('#foxco-line-limit').val();
        window.location.href = foxcoDebugViewer.ajax_url + 
            '?action=foxco_download_log&nonce=' + foxcoDebugViewer.nonce + 
            '&lines=' + lines;
    });
    
    // Filter checkboxes
    $('.foxco-debug-controls input[type="checkbox"]').on('change', function() {
        applyFilters();
    });
    
    // Line limit change
    $('#foxco-line-limit').on('change', function() {
        loadLog();
    });
    
    // Search
    let searchTimeout;
    $('#foxco-search').on('keyup', function() {
        clearTimeout(searchTimeout);
        currentSearchTerm = $(this).val().trim();
        
        searchTimeout = setTimeout(function() {
            applyFilters();
        }, 300);
    });
    
    function loadLog() {
        const lines = $('#foxco-line-limit').val();
        
        $.ajax({
            url: foxcoDebugViewer.ajax_url,
            type: 'POST',
            data: {
                action: 'foxco_get_log',
                nonce: foxcoDebugViewer.nonce,
                lines: lines
            },
            beforeSend: function() {
                $('#foxco-log-container').html('<div class="foxco-loading">Loading debug log...</div>');
            },
            success: function(response) {
                if (response.success) {
                    displayLog(response.data.content);
                } else {
                    $('#foxco-log-container').html('<div class="foxco-loading">Error loading log: ' + response.data + '</div>');
                }
            },
            error: function() {
                $('#foxco-log-container').html('<div class="foxco-loading">An error occurred while loading the log</div>');
            }
        });
    }
    
    function displayLog(content) {
        if (!content || content.trim() === '') {
            $('#foxco-log-container').html('<div class="foxco-loading">Log is empty</div>');
            return;
        }
        
        const lines = content.split('\n');
        let html = '';
        
        lines.forEach(function(line) {
            if (!line.trim()) return;
            
            let lineClass = 'log-line';
            
            if (line.match(/PHP Fatal error/i) || line.match(/PHP Parse error/i)) {
                lineClass += ' log-error';
            } else if (line.match(/PHP Warning/i)) {
                lineClass += ' log-warning';
            } else if (line.match(/PHP Notice/i) || line.match(/PHP Deprecated/i)) {
                lineClass += ' log-notice';
            }
            
            const escapedLine = $('<div>').text(line).html();
            html += '<div class="' + lineClass + '" data-type="' + lineClass + '">' + escapedLine + '</div>';
        });
        
        $('#foxco-log-container').html(html);
        
        // Scroll to bottom
        const container = document.getElementById('foxco-log-container');
        container.scrollTop = container.scrollHeight;
        
        applyFilters();
    }
    
    function applyFilters() {
        const showErrors = $('#foxco-filter-errors').is(':checked');
        const showWarnings = $('#foxco-filter-warnings').is(':checked');
        const showNotices = $('#foxco-filter-notices').is(':checked');
        
        $('.log-line').each(function() {
            const $line = $(this);
            let visible = true;
            
            // Type filters
            if ($line.hasClass('log-error') && !showErrors) {
                visible = false;
            } else if ($line.hasClass('log-warning') && !showWarnings) {
                visible = false;
            } else if ($line.hasClass('log-notice') && !showNotices) {
                visible = false;
            }
            
            // Search filter
            if (visible && currentSearchTerm) {
                const lineText = $line.text().toLowerCase();
                const searchLower = currentSearchTerm.toLowerCase();
                
                if (!lineText.includes(searchLower)) {
                    visible = false;
                } else {
                    // Highlight search term
                    const originalText = $line.text();
                    const regex = new RegExp('(' + escapeRegex(currentSearchTerm) + ')', 'gi');
                    const highlightedText = originalText.replace(regex, '<span class="highlight">$1</span>');
                    $line.html(highlightedText);
                }
            }
            
            if (visible) {
                $line.show();
            } else {
                $line.hide();
            }
        });
    }
    
    function escapeRegex(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }
});
